' MakePartitions for HexParty Game
' Rev 1.0.0 William M Leue, 24-Feb-2024

option default integer
option base 1
option angle degrees

' Constants
const NSIZES   = 3
const CSIZE    = 20
const WIDTH    = 2*CSIZE
const HEIGHT   = sqr(3)*CSIZE
const HSPACE   = 1.5*WIDTH
const MAXHPR   = 19
const CX       = mm.hres\2
const CY       = mm.vres\2

const MAX_NPART = 13

const MCHAN = 2
const ESC = 27

const HCOLOR  = rgb(100, 100, 255)

const UMSG_Y = 560
const UMSG_W = 300
const UMSG_H = 25
const BANN_Y = 1
const BANN_W = 400
const BANN_H = 50

' Keyboard commands
const PGUP  = 136
const PGDWN = 137
const KEND  = 135

' Globals
dim size   = 0
dim npart  = 0
dim lclick = 0
dim lbusy  = 0
dim mx     = 0
dim my     = 0
dim nrows(NSIZES)
dim nhpr(NSIZES, MAXHPR)
dim thexes(NSIZES)
dim maxnp(NSIZES)
dim board(2, 2)
dim sizenames$(NSIZES) = ("SMALL", "MEDIUM", "LARGE")
dim pallette(MAX_NPART)
dim pnum = 0
dim saved = 0
dim modified = 0
dim existing = 0
dim puzz_num = 0

' Main Program
open "debug.txt" for output as #1
InitMouse
ReadHexData
ReadPallette
cls
print "Need Instructions? (Y,N): ";
input "", a$
if LEFT$(UCASE$(a$), 1) = "Y" then
  ShowHelp
end if
do
  ChooseSize
  do
    DrawBoard size
    HandleEvents
  loop
loop
end

' Init Mouse and Cursor
sub InitMouse
  controller mouse open MCHAN, LeftClick
  gui cursor on
  settick 20, UpdateCursor
end sub

' Mouse Left Click ISR
sub LeftClick
  if not lbusy then
    lclick = 1
  end if
end sub

' Make cursor track the mouse
sub UpdateCursor
  mx = mouse(X)
  my = mouse(Y)
  gui cursor mx, my
end sub

' Read the Hex board data
sub ReadHexData
  local s, i, j, n
  for s = 1 to NSIZES
    read nrows(s)
    n = 0
    for i = 1 to nrows(s)
      read nhpr(s, i)
      inc n, nhpr(s, i)
    next i
    thexes(s) = n
  next s
  for s = 1 to NSIZES
    read maxnp(s)
  next s
  npart = 0
end sub

' Read the color pallette data and map colors
sub ReadPallette
  local i
  for i = 1 to MAX_NPART
    read pallette(i)
    map(i) = pallette(i)
  next i
  map set
end sub

' Choose a board size
sub ChooseSize
  local a$, p$, row, h
  cls
  do
    print "Choose Size (S)mall, (M)edium, or (L)arge: ";
    input "", a$
    p$ = LEFT$(UCASE$(a$), 1)
    select case p$
      case "S" : size = 1 : exit do
      case "M" : size = 2 : exit do
      case "L" : size = 3 : exit do
    end select
  loop
  erase board()
  dim board(nrows(size), MAXHPR)
  print "Work on (N)ew partition or (E)existing one?: ";
  input "", a$
  if LEFT$(UCASE$(a$), 1) = "E" then
    existing = 1
    print "Enter partition number: ";
    input "", a$
    puzz_num = val(a$)
    LoadPartition
  else
    modified = 1
    for row = 1 to nrows(size)
      for h = 1 to MAXHPR
        board(row, h) = 0
      next h
    next row
  end if
  pnum = 1
  ShowPartNum
end sub

' Display the current partition segment number at lower right
sub ShowPartNum
  local m$
  text mm.hres-1, mm.vres-1, space$(20), "RB"
  m$ = "Partition: " + str$(pnum)
  text mm.hres-1, mm.vres-1, m$, "RB"
end sub

' Draw the Game Board
sub DrawBoard size
  local row, h, x, y, n, pnum
  local r, g, b
  cls
  text 0, 0, "Press the END key after the grid is full to save your work.", "LT"
  gui cursor hide
  for row = 1 to nrows(size)
    n = nhpr(size, row)
    for h = 1 to n
      pnum = board(row, h)
      if pnum > 0 then
        c = map(pnum)
      else
        c = HCOLOR
      end if
      DrawCell row, h, c
    next h
  next row
  gui cursor show
end sub

' x, y is center of hexagon
sub DrawCell row, h, c
  local xv(6), yv(6)
  local n, p, x, y
  local m$
  n = nhpr(size, row)
  x = CX - (n-1)*0.5*HSPACE + (h-1)*HSPACE
  y = CY - (nrows(size)\2)*0.5*HEIGHT + (row-1)*0.5*HEIGHT
  xv(1) = x - CSIZE            : yv(1) = y
  xv(2) = x + CSIZE*cos(120)   : yv(2) = y - CSIZE*sin(120)
  xv(3) = x + CSIZE*cos(60)    : yv(3) = y - CSIZE*sin(60)
  xv(4) = x + CSIZE            : yv(4) = y
  xv(5) = x + CSIZE*cos(300)   : yv(5) = y - CSIZE*sin(300)
  xv(6) = x + CSIZE*cos(240)   : yv(6) = y - CSIZE*sin(240)
  polygon 6, xv(), yv(), rgb(white), c
  p = board(row, h)
  if p > 0 then
    text x, y, str$(p), "CM", 7,, rgb(black), -1
  end if
end sub

' Handle User events
sub HandleEvents
  local z$, cmd, c
  z$ = INKEY$
  c = rgb(red)
  do
    z$ = INKEY$
    if z$ <> "" then
      cmd = asc(UCASE$(z$))
      select case cmd
        case PGUP
          if pnum < maxnp(size) then
            inc pnum
            if pnum > npart then npart = pnum
            ShowPartNum
          end if
        case PGDWN
          if pnum > 1 then
            inc pnum, -1
            ShowPartNum
          end if
        case KEND
          if not CheckPartition() then
            Beep
          else
            SavePartition
            cls
            end
          end if
        case ESC
          if not modified then Quit
          if not saved then
            DrawUserMessage "You must press the END key to save your work before quitting!"
            continue do
          end if
          Quit
      end select
    end if
    if lclick then
      lbusy = 1
      HandleLeftClick
      lclick = 0
      lbusy = 0
    end if
  loop
end sub

' Quit the program
sub Quit
  settick 0, UpdateCursor
  gui cursor off
  controller mouse close
  cls
  end
end sub

' Handle Left Mouse Click
sub HandleLeftClick
  local row, h, m$
  GetClickedCell row, h
  text mm.hres-1, 0, space$(30), "RT"
  if (row >= 1) and (row <= nrows(size)) then
    if (h >= 1) and (h <= nhpr(size, row)) then
      m$ = "Row: " + str$(row) + ", h: " + str$(h)
      text mm.hres-1, 0, m$, "RT"
      if board(row, h) > 0 then
        board(row, h) = 0
        modified = 1
      else
        if GetNumCells(pnum) >= maxnp(size) then
          Beep
          DrawUserMessage "Maximum number of cells already assigned"
          exit sub
        end if
        board(row, h) = pnum
        modified = 1
    end if
    gui cursor hide
    if board(row, h) > 0 then
      DrawCell row, h, map(pnum)
    else
      DrawCell row, h, HCOLOR
    end if
    gui cursor show
  end if
end sub

' Convert mouse click coordinates to a grid (row, hex) location
sub GetClickedCell row, h
  local i, x, y, r, dx, dy
  for row = 1 to nrows(size)
    n = nhpr(size, row)
    x = CX - (n-1)*0.5*HSPACE
    y = CY - (nrows(size)\2)*0.5*HEIGHT + (row-1)*0.5*HEIGHT
    for h = 1 to n
      dx = mx-x : dy = my-y
      r = sqr(dx*dx + dy*dy)
      if r <= CSIZE then exit sub
      inc x, HSPACE
    next h
  next row
end sub

' Count the number of cells in a partition
function GetNumCells(p)
  local row, col, n
print #1, "GetNumCells(";p;")"
  n = 0
  for row = 1 to nrows(size)
    for col = 1 to nhpr(size, row)
      if board(row, col) = p then inc n
    next col
  next row
print #1, "  n: ";n
  GetNumCells = n
end function

' Beep for an error
sub Beep
  play tone 800, 800, 500
end sub

' Check the partition for completeness
' Returns 1 if complete, 0 if there are cells
' not assigned to a partition segment.
function CheckPartition()
  local row, h, n
  CheckPartition = 1
  for row = 1 to nrows(size)
    n = nhpr(size, row)
    for h = 1 to n
      if board(row, h) = 0 then
        CheckPartition = 0
        exit function
      end if
    next h
  next row
end function  

' Load Partition from disk
sub LoadPartition
  local p$, buf$, ps, row, h, solved
  p$ = "./PUZZLES/" + UCASE$(sizenames$(size)) + "/part" + str$(puzz_num) + ".prt"
  on error skip 1
  open p$ for input as #2
  if mm.errno <> 0 then
    cls
    print "Error opening file '";p$;"' for input: ";mm.errmsg$
    end
  end if
  line input #2, buf$
  ps = val(buf$)
  if ps <> size then
    cls
    print "Error : partition grid size does not match selected size"
    end
  end if
  line input #2, buf$
  npart = val(buf$)
  line input #2, buf$
  solved = val(buf$)
  for row = 1 to nrows(size)
    line input #2, buf$
    for h = 1 to nhpr(size, row)
      board(row, h) = val(field$(buf$, h, ","))
    next h
  next row
  close #2
  DrawBoard size
end sub

' Save the Partition to disk
sub SavePartition
  local np, n, path$, i, row, h, solved
  if existing then
    np = puzz_num
  else
    np = GetNextPartNum(size)
  end if
  path$ = "./PUZZLES/" + sizenames$(size) + "/part" + str$(np) + ".prt"
  on error skip 1
  open path$ for output as #2
  if mm.errno <> 0 then
    cls
    print "Error opening '";path$;" for output: ";mm.errmsg$
    end
  end if
  print #2, str$(size)
  print #2, str$(npart)
  solved = 0
  print #2, str$(solved)
  for row = 1 to nrows(size)
    n = nhpr(size, row)
    for h = 1 to n
      print #2, str$(board(row, h)) + ",";
    next h
    print #2, ""
  next row
  close #2
end sub

' Scan the PUZZLES/<size> directory to find the largest partition number
' and return that number plus 1.
function GetNextPartNum(size)
  local p$, f$, n, r$, fl, maxn
  n = 0 : maxn = 0
  p$ = "./PUZZLES/" + sizenames$(size) + "/*"
  f$ = DIR$(p$, FILE)
  do while f$ <> ""
    fl = len(f$)
    r$ = LEFT$(f$, fl-4)
    r$ = MID$(r$, 5)
    n = val(r$)
    if n > maxn then maxn = n
    f$ = DIR$()
  loop
  GetNextPartNum = maxn+1
end function

' Draw a Banner at screen top
' b$: banner text
' f: font number
' fc: font color
sub DrawBanner b$, f, fc, bc
  if bc <> 0 then
    box mm.hres\2-BANN_W\2, BANN_Y, BANN_W, BANN_H,, bc, bc
  end if
  text mm.hres\2, BANN_Y+BANN_H\2, b$, "CM", f,, fc, -1
end sub

' Draw a User Message at screen bottom (auto-erase)
sub DrawUserMessage m$
  text mm.hres\2, UMSG_Y+UMSG_H\2, m$, "CM",,, rgb(red)
  pause 2000
  text mm.hres\2, UMSG_Y+UMSG_H\2, space$(60), "CM",,, rgb(red)
end sub

' Help
sub ShowHelp
  local z$ = INKEY$
  cls
  text mm.hres\2, 0, "Help for MakePartition", "CT", 4,, rgb(green)
  print @(0, 30) ""
  print "MakePartition allows you to create new puzzles for HexParty, or edit"
  print "previously-created puzzles."
  print ""
  print "To begin, you will be asked to choose a puzzle size: SMALL, MEDIUM, or LARGE."
  print "Next, you will be asked if you want to create a (N)ew puzzle or work on an"
  print "(E)xisting one. If you choose (N)ew, you will see a blank grid appear."
  print ""
  print "If you choose to work on an existing puzzle, you will be informed"
  print "how many puzzles currently exist in the PUZZLES/<size> folder. Puzzles are numbered"
  print "serially, so you need to answer with the number 1, 2., 3... of the specific"
  print "puzzle you want to work on. After you choose, you will see the fully-assembled"
  print "puzzle appear in the grid."
  print ""
  print "At the bottom right-hand corner of the screen, you will see the current partition"
  print "number. You can increase it or decrease it by pressing the PAGEUP or PAGEDOWN keys."
  print "The maximum partition number depends on the grid size: 6, 8, or 13 for the SMALL,"
  print "MEDIUM, and LARGE grids respectively."
  print ""
  print "To set or clear the partition number for a hexagonal cell on the grid, left-click"
  print "the cell. It will be changed to the color for the current partition number and the"
  print "number will appear in the cell. Note that partitions do NOT have to be continuous:"
  print "that is, you can have disconnected cells in a partition.  A partition can have from"
  print "1 to 13 cells."
  print ""
  print "Continue selecting cells and assigning partition numbers until EVERY cell has been"
  print "assigned. You cannot save the puzzle until every cell has a partition number."
  print "Press the END key to save your work to a puzzle file. You do not need to name the"
  print "file: it will be automatically saved to the './PUZZLES/<size> folder and named with"
  print "the correct sequence number."
  text mm.hres\2, mm.vres-5, "Press Any Key to Continue", "CB"
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub
  
' return a uniformly distributed random integer in the specified closed range
function RandInt(a as integer, b as integer)
  local integer v, c
  c = b-a+1
  do
    v = a + (b-a+2)*rnd()
    if v >= a and v <= b then exit do
  loop
  RandInt = v
end function

' Hex data: each row begins with number of rows in board,
' followed by number of hexes per row.
data  7, 2, 1, 2, 3, 2, 1, 2
data 13, 3, 2, 3, 4, 3, 4, 5, 4, 3, 4, 3, 2, 3
data 19, 4, 3, 4, 5, 4, 5, 6, 5, 6, 7, 6, 5, 6, 5, 4, 5, 4, 3, 4

' Max number of partitions per size
data 6, 8, 13

' Color Pallette
data rgb(red), rgb(green), rgb(blue), rgb(yellow), rgb(cyan), rgb(magenta)
data rgb(179, 18, 84), rgb(128, 65, 0), rgb(255, 142, 0), rgb(255, 178, 178)
data rgb(38, 128, 49), rgb(127, 127, 255), rgb(142, 25, 255)


